"""
Sistema de Persistência de Estado para Threshold Module
"""
import json
import os
import threading
import time
from datetime import datetime
from typing import Dict, Any, Optional


class StatePersistence:
    def __init__(self, state_file: str = None):
        """
        Inicializa o sistema de persistência de estado
        
        Args:
            state_file: Caminho para o arquivo de estado
        """
        if state_file is None:
            # Busca o arquivo na raiz do projeto
            current_dir = os.path.dirname(os.path.abspath(__file__))
            self.state_file = os.path.join(current_dir, "threshold_state.json")
        else:
            self.state_file = state_file
        
        self.auto_save_enabled = True
        self.auto_save_interval = 60  # segundos
        self.last_save_time = None
        self.save_thread = None
        self.running = False
        
        # Cria backup do estado anterior se existir
        self._backup_previous_state()
    
    def _backup_previous_state(self):
        """Cria backup do estado anterior"""
        if os.path.exists(self.state_file):
            backup_file = f"{self.state_file}.backup"
            try:
                import shutil
                shutil.copy2(self.state_file, backup_file)
                print(f"📋 Backup do estado anterior criado: {backup_file}")
            except Exception as e:
                print(f"⚠️ Erro ao criar backup do estado: {e}")
    
    def load_state(self) -> Dict[str, Any]:
        """
        Carrega o estado do programa
        
        Returns:
            Dicionário com o estado do programa
        """
        try:
            if os.path.exists(self.state_file):
                with open(self.state_file, 'r', encoding='utf-8') as f:
                    state = json.load(f)
                    print(f"📂 Estado carregado com sucesso: {self.state_file}")
                    return state
            else:
                print("📂 Nenhum estado anterior encontrado, iniciando com estado padrão")
                return self._get_default_state()
        except (json.JSONDecodeError, IOError) as e:
            print(f"⚠️ Erro ao carregar estado: {e}")
            # Tenta carregar backup se disponível
            backup_file = f"{self.state_file}.backup"
            if os.path.exists(backup_file):
                try:
                    with open(backup_file, 'r', encoding='utf-8') as f:
                        state = json.load(f)
                        print(f"📂 Estado restaurado do backup: {backup_file}")
                        return state
                except Exception as backup_error:
                    print(f"⚠️ Erro ao carregar backup: {backup_error}")
            
            return self._get_default_state()
    
    def _get_default_state(self) -> Dict[str, Any]:
        """
        Retorna o estado padrão do programa
        
        Returns:
            Estado padrão
        """
        return {
            "interface_config": {
                "min_freq": "840",
                "max_freq": "960",
                "max_power": "25",
                "freq_step": "5",
                "power_step": "1.0",
                "distance": "1.0",
                "attenuator": "0.0",
                "description": "",
                "graph1_type": "Module Power (dBm)",
                "graph2_type": "Module Power (dBm)"
            },
            "tags": [],
            "test_history": [],
            "projects": [],
            "selected_tags": [],
            "selected_tests": [],
            "graph_settings": {
                "show_average": False,
                "auto_scale": True
            },
            "last_session": {
                "timestamp": datetime.now().isoformat(),
                "version": "1.0"
            }
        }
    
    def save_state(self, state_data: Dict[str, Any]) -> bool:
        """
        Salva o estado do programa
        
        Args:
            state_data: Dados do estado a serem salvos
            
        Returns:
            True se salvou com sucesso, False caso contrário
        """
        try:
            # Atualiza timestamp da sessão
            state_data["last_session"]["timestamp"] = datetime.now().isoformat()
            
            # Cria backup temporário antes de salvar
            temp_file = self.state_file + ".tmp"
            with open(temp_file, 'w', encoding='utf-8') as f:
                json.dump(state_data, f, indent=4, ensure_ascii=False)
            
            # Move o arquivo temporário para o arquivo final
            import shutil
            shutil.move(temp_file, self.state_file)
            
            self.last_save_time = datetime.now()
            print(f"💾 Estado salvo com sucesso: {self.state_file}")
            return True
        except Exception as e:
            print(f"❌ Erro ao salvar estado: {e}")
            return False
    
    def start_auto_save(self, interval_seconds: int = 60):
        """
        Inicia o salvamento automático
        
        Args:
            interval_seconds: Intervalo em segundos entre salvamentos
        """
        if self.running:
            return
        
        self.auto_save_interval = interval_seconds
        self.running = True
        self.save_thread = threading.Thread(target=self._auto_save_loop, daemon=True)
        self.save_thread.start()
        print(f"🔄 Auto-save iniciado (intervalo: {interval_seconds} segundos)")
    
    def stop_auto_save(self):
        """Para o salvamento automático"""
        self.running = False
        if self.save_thread:
            self.save_thread.join(timeout=1)
        print("⏹ Auto-save parado")
    
    def _auto_save_loop(self):
        """Loop de salvamento automático"""
        while self.running:
            try:
                time.sleep(self.auto_save_interval)
                if self.running and self.auto_save_enabled:
                    # O estado será salvo pelo callback do módulo principal
                    pass
            except Exception as e:
                print(f"⚠️ Erro no auto-save: {e}")
    
    def get_state_info(self) -> Dict[str, Any]:
        """
        Retorna informações sobre o estado atual
        
        Returns:
            Informações do estado
        """
        try:
            if os.path.exists(self.state_file):
                stat = os.stat(self.state_file)
                return {
                    "file_exists": True,
                    "file_size": stat.st_size,
                    "last_modified": datetime.fromtimestamp(stat.st_mtime).isoformat(),
                    "auto_save_enabled": self.auto_save_enabled,
                    "last_save_time": self.last_save_time.isoformat() if self.last_save_time else None
                }
            else:
                return {
                    "file_exists": False,
                    "auto_save_enabled": self.auto_save_enabled,
                    "last_save_time": None
                }
        except Exception as e:
            return {
                "error": str(e),
                "auto_save_enabled": self.auto_save_enabled
            }


